const profilesBox = document.getElementById("profiles");
const addBtn = document.getElementById("add");

const whitelistEl = document.getElementById("whitelist");
const blacklistEl = document.getElementById("blacklist");
const saveSitesBtn = document.getElementById("saveSites");

let profiles = [];
let whitelist = [];
let blacklist = [];

chrome.storage.sync.get(["profiles", "whitelist", "blacklist"], (data) => {
    profiles = data.profiles || [];
    whitelist = data.whitelist || [];
    blacklist = data.blacklist || [];

    whitelistEl.value = (whitelist || []).join("\n");
    blacklistEl.value = (blacklist || []).join("\n");

    renderProfiles();
});

function saveProfiles() {
    chrome.storage.sync.set({ profiles });
}

function normalizeHosts(text) {
    return text
        .split("\n")
        .map(s => s.trim())
        .filter(Boolean)
        .map(s => s.replace(/^https?:\/\//, "").replace(/\/.*$/, "")); // garde hostname
}

saveSitesBtn.addEventListener("click", () => {
    whitelist = normalizeHosts(whitelistEl.value);
    blacklist = normalizeHosts(blacklistEl.value);
    chrome.storage.sync.set({ whitelist, blacklist });
});

// --- PACKS ---
async function loadPack(id) {
    const url = chrome.runtime.getURL(`packs/${id}.json`);
    const res = await fetch(url);
    return await res.json();
}

document.querySelectorAll("#packs button").forEach(btn => {
    btn.addEventListener("click", async () => {
        const packId = btn.dataset.pack;
        const pack = await loadPack(packId);

        if (profiles.some(p => p.id === pack.id)) {
            alert("Pack déjà installé");
            return;
        }

        profiles.push({
            id: pack.id,
            name: pack.name,
            enabled: true,
            aliases: pack.aliases || [],
            keywords: pack.keywords || { fort: [], moyen: [], faible: [] }
        });
        chrome.storage.sync.get(["stats"], d => {
            const stats = d.stats || {};
            stats.packsAdded = stats.packsAdded || {};
            stats.packsAdded[pack.id] = (stats.packsAdded[pack.id] || 0) + 1;
            chrome.storage.sync.set({ stats });
        });
        saveProfiles();
        renderProfiles();
    });
});

// --- PROFILES UI ---
function renderProfiles() {
    profilesBox.innerHTML = "";

    profiles.forEach((p, i) => {
        const div = document.createElement("div");
        div.className = "profile";

        div.innerHTML = `
      <div class="row">
        <label class="row small">
          <input type="checkbox" data-i="${i}" data-k="enabled" ${p.enabled ? "checked" : ""}/>
          <span>Actif</span>
        </label>

        <div style="flex:1; min-width:220px;">
          <label>Nom du profil</label>
          <input type="text" data-i="${i}" data-k="name" value="${escapeHtml(p.name)}"/>
        </div>
      </div>

      <label>Alias (séparés par virgule)</label>
      <input type="text" data-i="${i}" data-k="aliases" value="${escapeHtml((p.aliases || []).join(", "))}"/>

      <label>Mots forts (+5)</label>
      <input type="text" data-i="${i}" data-k="fort" value="${escapeHtml((p.keywords?.fort || []).join(", "))}"/>

      <label>Mots moyens (+3)</label>
      <input type="text" data-i="${i}" data-k="moyen" value="${escapeHtml((p.keywords?.moyen || []).join(", "))}"/>

      <label>Mots faibles (+1)</label>
      <input type="text" data-i="${i}" data-k="faible" value="${escapeHtml((p.keywords?.faible || []).join(", "))}"/>

      <div class="actions">
        <button class="btn danger" data-i="${i}" data-action="delete">Supprimer</button>
      </div>
    `;

        profilesBox.appendChild(div);
    });

    // listeners inputs
    profilesBox.querySelectorAll("input[type='text']").forEach(input => {
        input.addEventListener("change", (e) => {
            const i = Number(e.target.dataset.i);
            const k = e.target.dataset.k;
            const val = e.target.value.trim();

            if (k === "name") profiles[i].name = val;

            if (k === "aliases") {
                profiles[i].aliases = val.split(",").map(v => v.trim()).filter(Boolean);
            }

            if (k === "fort" || k === "moyen" || k === "faible") {
                profiles[i].keywords = profiles[i].keywords || { fort: [], moyen: [], faible: [] };
                profiles[i].keywords[k] = val.split(",").map(v => v.trim()).filter(Boolean);
            }

            saveProfiles();
        });
    });

    // listeners checkboxes
    profilesBox.querySelectorAll("input[type='checkbox']").forEach(cb => {
        cb.addEventListener("change", (e) => {
            const i = Number(e.target.dataset.i);
            profiles[i].enabled = e.target.checked;
            saveProfiles();
        });
    });

    // delete
    profilesBox.querySelectorAll("button[data-action='delete']").forEach(btn => {
        btn.addEventListener("click", () => {
            const i = Number(btn.dataset.i);
            const name = profiles[i]?.name || "ce profil";
            if (!confirm(`Supprimer ${name} ?`)) return;

            profiles.splice(i, 1);
            saveProfiles();
            renderProfiles();
        });
    });
}

addBtn.addEventListener("click", () => {
    profiles.push({
        id: crypto.randomUUID(),
        name: "Nouveau profil",
        enabled: true,
        aliases: [],
        keywords: { fort: [], moyen: [], faible: [] }
    });
    chrome.storage.sync.get(["stats"], d => {
        const stats = d.stats || {};
        stats.packsAdded = stats.packsAdded || {};
        stats.packsAdded[pack.id] = (stats.packsAdded[pack.id] || 0) + 1;
        chrome.storage.sync.set({ stats });
    });
    saveProfiles();
    renderProfiles();
});

// petite protection XSS (options UI)
function escapeHtml(str) {
    return String(str || "")
        .replaceAll("&", "&amp;")
        .replaceAll("<", "&lt;")
        .replaceAll(">", "&gt;")
        .replaceAll('"', "&quot;")
        .replaceAll("'", "&#039;");
}

const statsBox = document.getElementById("stats");
chrome.storage.sync.get(["stats"], d => {
    statsBox.textContent = JSON.stringify(d.stats || {}, null, 2);
});

